#pragma once

#include "PMC_Enums.h"

#define MAXREALXBOTS 100

namespace PMC
{	
	/// <summary>
	/// Return info for the FindPMCs function
	/// </summary>
	struct FindPMCsRtn
	{
		/// <summary>
		/// PMC command return for the FindPMCs command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;		
		/// <summary>
		/// The number PMCs found on EtherCAT 
		/// </summary>
		PMC_UINT numPMCsFound = 0;		
		/// <summary>
		/// The IO indexes of the PMCs found on EtherCAT
		/// </summary>
		PMC_UINT PMCIOIndexes[256];
	};

	/// <summary>
	/// return struct for motion commands, includes PMC return + travel time
	/// </summary>
	struct MotionRtn
	{
		/// <summary>
		/// PMC return for the motion command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// travelling time for the command, in seconds
		/// </summary>
		PMC_REAL TravelTimeSecs = 0;
	};

	/// <summary>
	/// Return information for PMC border status
	/// </summary>
	struct BorderStatusRtn
	{
		/// <summary>
		/// PMC command return for the border status query command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// The status of the border
		/// </summary>
		BORDERSTATUS BorderStatus = BORDERSTATUS::DISCONNECTED;
	};

	/// <summary>
	/// Return information for getting new xbots at PMC borders
	/// </summary>
	struct BorderNewXbotRtn
	{
		/// <summary>
		/// PMC command return for the border status query command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// Number of new received XBOTs
		/// </summary>
		PMC_USINT XBotCount = 0;
		/// <summary>
		/// The current IDs of the received XBOTs
		/// </summary>
		PMC_USINT XBotIDs[100];
		/// <summary>
		/// The border ID where each xbot was received from
		/// </summary>
		PMC_USINT BorderIDs[100];
	};

	/// <summary>
	///  structure for WaitUntil command
	/// </summary>
	struct WaitUntilTriggerParams
	{
		/// <summary>
		/// When trigger source is set to time delay, this is the delay duration, in seconds
		/// </summary>
		PMC_REAL delaySecs = 0;     //trigger delay time(s)
		/// <summary>
		/// When trigger source is set to Digital Input (external or fieldbus), this is digital input channel ID.
		/// </summary>
		PMC_USINT externalOrFBInputChannelID = 0;        //external digital input channel
		/// <summary>
		/// When trigger source is set to Digital Input (external or fieldbus), the edge type or value that will trigger the event
		/// </summary>
		TRIGGEREDGETYPE edgeType;      //0: rising edge;1: falling edge;
		/// <summary>
		/// When trigger source is set to Command Label or Displacement, this is the ID of the XBOT to act as the trigger source 
		/// </summary>
		PMC_USINT triggerXbotID = 0;     //trigger xbot ID
		/// <summary>
		/// When trigger source is set to Command Label, this is Command Label to trigger the event
		/// </summary>
		PMC_USINT triggerCmdLabel = 0;   //trigger command label

		/// <summary>
		/// For specifying hte details of a command label trigger.
		/// 0 = motion command label; 1 = Run Macro command label
		/// </summary>
		TRIGGERCMDTYPE triggerCmdType;   //trigger command label
		/// <summary>
		/// When trigger source is set to Command Label,
		/// CMD_Start means the event will trigger when the command first starts executing, 
		/// CMD_FINISH = after command has finished executing, 
		/// CMD_EXECUTING = at any point during command execution
		/// </summary>
		TRIGGERCMDLABELTYPE CmdLabelTriggerType;    //0: command start;1: command end;
		/// <summary>
		/// When the trigger source is set to Displacement, the statement AX+BY vs Threshold is evaluated.
		/// GREATER_THAN means event triggers when AX+BY > Threshold, if is X_ONLY mode, then X > Threshold, in Y_ONLY mode, Y > Threshold. 
		/// LESS_THAN triggers when AX+BY less than Threshold. 
		/// POSITIVE_CROSS triggers when AX+BY is first less than Threshold, then later greater than Threshold. 
		/// NEGATIVE_CROSS triggers when AX+BY is first greater than Threshold, then later less than Threshold. 
		/// </summary>
		TRIGGERDISPLACEMENTTYPE displacementTriggerType;    //0: greater than;1: less than;
		/// <summary>
		/// When the trigger source is set to Displacement, this will determine if only the X position is evaluated,
		/// or if only the Y position is evaluated, or if the position is evaluated as AX+BY
		/// </summary>
		TRIGGERDISPLACEMENTMODE displacementTriggerMode;    //0: greater than;1: less than;
		/// <summary>
		/// When the trigger source is set to Displacement, and the trigger mode is set to AX+BY, 
		/// this will specify the A parameter
		/// </summary>
		PMC_REAL lineParamAx = 0;
		/// <summary>
		/// When the trigger source is set to Displacement, and the trigger mode is set to AX+BY, 
		/// this will specify the B parameter
		/// </summary>
		PMC_REAL lineParamBy = 0;
		/// <summary>
		/// When the trigger source is set to Displacement, this will set the threshold to be compared to
		/// </summary>
		PMC_REAL displacementThresholdMeters = 0;
	};

	/// <summary>
	/// full XBot status, including it's position and state
	/// </summary>
	struct XBotStatusRtn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// XBot state
		/// </summary>
		XBOTSTATE State = XBOTSTATE::XBOT_UNDETECTED;		
		/// <summary>
		/// Command Label@Motion State
		/// </summary>
		PMC_UINT CmdLb = 0;		
		/// <summary>
		/// 0: Not Force Mode;1: X force mode;2: Y force mode;4: Z force mode;3: XY force mode
		/// </summary>
		PMC_USINT FMState = 0;		
		/// <summary>
		/// 0: Not linked to any xbot; >0: start Xbot ID
		/// </summary>
		PMC_USINT StarXID = 0;
		/// <summary>
		/// 0: Not bond to any group; >0: group ID
		/// </summary>
		PMC_USINT GroupID = 0;
		/// <summary>
		/// number of commands in motion buffer
		/// </summary>
		PMC_UINT nCmds = 0;		
		/// <summary>
		/// is the xbot buffer blocked
		/// </summary>
		PMC_BOOL IsBlocked = false;		
		/// <summary>
		/// is the xbot motion paused
		/// </summary>
		PMC_BOOL IsPaused = false;		
		/// <summary>
		/// xbot X position in meters, force in Newtons
		/// </summary>
		PMC_REAL PosX = 0;		
		/// <summary>
		/// xbot Y position in meters, force in Newtons
		/// </summary>
		PMC_REAL PosY = 0;		
		/// <summary>
		/// xbot Z position in meters, force in Newtons
		/// </summary>
		PMC_REAL PosZ = 0;		
		/// <summary>
		/// xbot rotation about the X axis in radians, torque in Newton*Meters 
		/// </summary>
		PMC_REAL PosRx = 0;		
		/// <summary>
		/// xbot rotation about the Y axis in radians, torque in Newton*Meters 
		/// </summary>
		PMC_REAL PosRy = 0;		
		/// <summary>
		/// xbot rotation about the Z axis in radians, torque in Newton*Meters 
		/// </summary>
		PMC_REAL PosRz = 0;		
		/// <summary>
		/// XBot stereotype ID
		/// </summary>
		PMC_USINT StereotypeID = 0;		
		/// <summary>
		/// XBot Type (0 = M3-06,2 = M3-08,4 = M3-09X,5 = M3-09Y,6 = M3-10,8 = M3-11X,9 = M3-11Y,12 = M3-12,14 = M3-13)
		/// </summary>
		PMC_USINT MoverType = 0;
		/// <summary>
		/// >0, stream ID that xbot is currently following
		/// </summary>
		PMC_UINT StreamID = 0;		
		/// <summary>
		/// current part state set using XbotPartCtrl
		/// </summary>
		PMC_USINT PartState = 0;
	};

	/// <summary>
	/// full return for the Get Flyway Status Command. Includes various temperature measurements and power consumption
	/// </summary>
	struct FlywayPhysicalStatus
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// power consumption, in watts
		/// </summary>
		PMC_REAL powerConsumptionW = 0;
		/// <summary>
		/// CPU temperature, in celsius
		/// </summary>
		PMC_REAL cpuTempC = 0;
		/// <summary>
		/// amplifier temperature, in celcius
		/// </summary>
		PMC_REAL amplifierTempC = 0;
		/// <summary>
		/// motor temperature, in celsius
		/// </summary>
		PMC_REAL motorTempC = 0;
	};

	/// <summary>
	/// a single XBOT's payload options
	/// </summary>
	struct XBotPayloadSettings
	{
		/// <summary>
		/// The weight of the payload, in kg
		/// </summary>
		PMC_REAL payloadkg = 0;
		/// <summary>
		/// the payload's center of gravity, in meters
		/// </summary>
		PMC_REAL payloadCGHeightm = 0;
		/// <summary>
		/// Payload X Dimensions(meters)
		/// </summary>
		PMC_REAL payloadDimensionXm = 0;
		/// <summary>
		/// Payload Y Dimensions(meters)
		/// </summary>
		PMC_REAL payloadDimensionYm = 0;
	};

	/// <summary>
	/// defines a single mover property
	/// </summary>
	struct MoverProperty
	{
		/// <summary>
		/// ID of the mover to set the property for
		/// </summary>
		PMC_USINT moverID = 0;
		/// <summary>
		/// the index of the mover property
		/// </summary>
		MOVERPROPERTY propertyID;
		/// <summary>
		/// the value of the mover property
		/// </summary>
		PMC_REAL propertyValue = 0;
	};

	/// <summary>
	/// full return for the Get Mover Property Command. Contains 8 mover properties
	/// </summary>
	struct XbotPropertyReturn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;		
		/// <summary>
		/// xbot type (0 = M3-06,2 = M3-08,4 = M3-09X,5 = M3-09Y,6 = M3-10,8 = M3-11X,9 = M3-11Y,12 = M3-12,14 = M3-13)
		/// </summary>
		MOVERTYPE MoverType;		
		/// <summary>
		/// payload weight in kilograms
		/// </summary>
		PMC_REAL PayloadW;		
		/// <summary>
		/// payload center of gravity in meters
		/// </summary>
		PMC_REAL PayloadCG;		
		/// <summary>
		/// mover X dimension in meters
		/// </summary>
		PMC_REAL XDim;		
		/// <summary>
		/// mover X dimension in meters
		/// </summary>
		PMC_REAL YDim;		
		/// <summary>
		/// acceleration limit in meters/second^2
		/// </summary>
		PMC_REAL accLimit;
	};

	/// <summary>
	/// full return for the Get Zone Status Command. Contains Zone information
	/// </summary>
	struct ZoneStatusReturn
	{
		/// <summary>
		/// return value for checking the zone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// Zone State: 0-undefined, 1-active unfenced, 2-deactivating, 3-loading, 4-activating, 5-active fenced
		/// </summary>
		ZONESTATE State;
		/// <summary>
		/// Number of XBots inside the zone
		/// </summary>
		PMC_USINT NumXbots = 0;
		/// <summary>
		/// IDs of xbots in the zone
		/// </summary>
		PMC_USINT XbotIDs[39];
	};

	/// <summary>
	/// full return for the Get Zone Status Extended Command. Contains Extended Zone information
	/// </summary>
	struct ZoneStatusExReturn
	{
		/// <summary>
		/// return value for checking the zone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// Zone State: 0-undefined, 1-active unfenced, 2-deactivating, 3-loading, 4-activating, 5-active fenced
		/// </summary>
		ZONESTATE State;
		/// <summary>
		/// Number of XBots inside the zone
		/// </summary>
		PMC_USINT NumXbotsInZone = 0;
		/// <summary>
		/// IDs of xbots in the zone
		/// </summary>
		PMC_USINT XbotIDsInZone[32];
		/// <summary>
		/// number of xbots on boundary of zone
		/// </summary>
		PMC_USINT NumXbotsOnBoundary = 0;
		/// <summary>
		/// xbot IDs of xbots on boundary
		/// </summary>
		PMC_USINT XbotIDsOnBoundary[32];
		/// <summary>
		/// 0-override with ratio,1-override with absolute speed and acceleration limits
		/// </summary>
		PMC_USINT OverrideType;
		/// <summary>
		/// multiplicative factor or absolute limit for xbot speed in the zone (0-2 or m/s)
		/// </summary>
		PMC_REAL SpeedOverrideFactor;
		/// <summary>
		/// multiplicative factor or absolute limit for xbot acceleration in the zone (0.001-2 or m/s^2)
		/// </summary>
		PMC_REAL AccOverrideFactor;
	};
		
	/// <summary>
	/// Full return from the GetPMCWarnings command
	/// </summary>
	struct WarningsReturn
	{
		/// <summary>
		/// return value for checking the zone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;		
		/// <summary>
		/// Number of warnings
		/// </summary>
		PMC_USINT nWarnings = 0;
		/// <summary>
		/// the warning codes
		/// </summary>
		PMC_UDINT Warnings[100];
	};

	/// <summary>
	/// full return for the Get Star Status Command. Contains star information
	/// </summary>
	struct StarStatusReturn
	{
		/// <summary>
		/// return value for checking the zone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;		
		/// <summary>
		/// 0: not defined, 1: defined, 2: error
		/// </summary>
		PMC_USINT WheelState = 0;		
		/// <summary>
		/// 0: not clear, 1: clear
		/// </summary>
		PMC_USINT EntranceState = 0;
		/// <summary>
		/// 0: no xbot in exit, 1: xbot in exit
		/// </summary>
		PMC_USINT ExitState = 0;
		/// <summary>
		/// Exit xbot ID
		/// </summary>
		PMC_USINT ExitXID = 0;
		/// <summary>
		/// Xbot start X coordinate in meters
		/// </summary>
		PMC_REAL StartX = 0;		
		/// <summary>
		/// Xbot start Y coordinate in meters
		/// </summary>
		PMC_REAL StartY = 0;		
		/// <summary>
		/// Xbot end X coordinate in meters
		/// </summary>
		PMC_REAL EndX = 0;		
		/// <summary>
		/// Xbot end Y coordinate in meters
		/// </summary>
		PMC_REAL EndY = 0;
	};

	/// <summary>
	/// mover stereotype definition
	/// </summary>
	struct MoverStereotypeData
	{
		/// <summary>
		/// The control performance level[0-3], 0 = most conservative level, 3 = most aggresive level
		/// </summary>
		PMC_USINT performanceLevel = 0;
		/// <summary>
		/// The weight of the payload, in kg
		/// </summary>
		PMC_REAL payloadkg = 0;
		/// <summary>
		/// Payload size from center of the XBOT, in +X direction (meters) 
		/// </summary>
		PMC_REAL payloadPositiveXmFromXBOTCenter = 0;
		/// <summary>
		/// Payload size from center of the XBOT, in -X direction (meters)
		/// </summary>
		PMC_REAL payloadNegativeXmFromXBOTCenter = 0;
		/// <summary>
		/// Payload size from center of the XBOT, in +Y direction (meters) 
		/// </summary>
		PMC_REAL payloadPositiveYmFromXBOTCenter = 0;
		/// <summary>
		/// Payload size from center of the XBOT, in -Y direction (meters) 
		/// </summary>
		PMC_REAL payloadNegativeYmFromXBOTCenter = 0;
		/// <summary>
		/// the payload's center of gravity in the x direction, in meters
		/// </summary>
		PMC_REAL payloadCGXm = 0;
		/// <summary>
		/// the payload's center of gravity in the y direction, in meters
		/// </summary>
		PMC_REAL payloadCGYm = 0;
		/// <summary>
		/// the payload's center of gravity in the z direction, in meters
		/// </summary>
		PMC_REAL payloadCGZm = 0;		
		/// <summary>
		/// Acceleration limit during emergency stop (m/s^2)
		/// </summary>
		PMC_REAL EmergencyStopDecel = 0;
	};

	/// <summary>
	/// full return for the Get Mover Stereotype Command. 
	/// </summary>	
	struct MoverStereotypeDefinitionReturn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// mover stereotype data
		/// </summary>
		MoverStereotypeData StereotypeData;
	};

	/// <summary>
	/// full return for the read assigned mover stereotype Command. 
	/// </summary>	
	struct ReadAssignedStereotypeReturn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// ID of the assigned mover stereotype
		/// </summary>
		PMC_USINT AssignedStereotypeID = 0;
	};

	/// <summary>
	/// full return for the Get Mover Property Command. Contains 8 mover properties
	/// </summary>	
	struct PayloadWeighingReturn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// Measured weight, in kg
		/// </summary>
		PMC_REAL WeightKG = 0;
	};

	/// <summary>
	/// return struct for get payload settings commands, includes PMC return + PayloadSettings
	/// </summary>	
	struct PayloadRtn
	{
		/// <summary>
		/// PMC return for the motion command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// an xbot's payload settings
		/// </summary>
		PMC_REAL Payload;
	};

	/// <summary>
	/// full return of the motion buffer command
	/// </summary>
	struct MotionBufferStatusRtn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// true is xbot buffer is currently blocked, false is buffer is unblocked
		/// </summary>
		PMC_BOOL State = false;
		/// <summary>
		/// number of motions commands stored in the xbot's buffer. If xbot ID = 0, the reply will be 0
		/// </summary>
		PMC_UINT nCmds = 0;
		/// <summary>
		/// command label of the first command in the buffer, it is the next command that will be executed by the xbot.
		/// </summary>
		PMC_UINT nextCmdLb = 0;
		/// <summary>
		/// command label of the last command in the buffer, this is the command that is most recently added to the motion buffer.
		/// </summary>
		PMC_UINT newCmdLb = 0;
	};

	/// <summary>
	/// a single XBOT's motion macro status
	/// </summary>
	struct MotionMacroStatus
	{
		/// <summary>
		/// 0 = macro is not saved (cannot be run); 2 = macro is saved (ready to run)
		/// </summary>
		PMC_USINT macroState = 0;
		/// <summary>
		/// the number of commands stored in this macro
		/// </summary>
		PMC_USINT storedCommandsCount = 0;
	};

	/// <summary>
	/// full return of the motion macro
	/// </summary>	
	struct MotionMacroReturn
	{
		/// <summary>
		/// return value of the motion macro command
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// the status of the motion macro
		/// </summary>
		MotionMacroStatus motionMacroStatus;
	};

	/// <summary>
	/// full return for the group control command
	/// </summary>	
	struct GroupStatusRtn
	{
		/// <summary>
		/// return value for checking the xbot status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// 0 = group is disconnected, 1 = group is connected
		/// </summary>
		PMC_USINT groupStatus = 0;
		/// <summary>
		/// number of xbots in this group
		/// </summary>
		PMC_USINT groupMembersCount = 0;
		/// <summary>
		/// the IDs of the xbots in this group
		/// </summary>
		PMC_USINT groupMembersID[32];
	};

	/// <summary>
	/// information required to setup 1 axis of the slave xbot's cam trajectory. 
	/// </summary>
	struct CamAxisDataClass
	{
		/// <summary>
		/// ID of the master XBOT
		/// </summary>
		PMC_USINT MasterXID = 0;
		/// <summary>
		/// Axis of the master XBOT to track
		/// </summary>
		AXISNAMES MasterAxisID = AXISNAMES::X_1;
		/// <summary>
		/// Axis of the slave XBOT used to follow the CAM
		/// </summary>
		AXISNAMES SlaveAxisID = AXISNAMES::X_1;
		/// <summary>
		/// ID of the Cam
		/// </summary>
		PMC_USINT CamID = 128;
	};

	/// <summary>
	/// information required to setup 1 axis of the slave xbot's cam trajectory. 
	/// </summary>
	struct CamAxisDataClassExtended
	{
		/// <summary>
		/// ID of the master XBOT
		/// </summary>
		PMC_USINT MasterXID = 0;
		/// <summary>
		/// Axis of the master XBOT to track
		/// </summary>
		AXISNAMES MasterAxisID = AXISNAMES::X_1;
		/// <summary>
		/// Axis of the slave XBOT used to follow the CAM
		/// </summary>
		AXISNAMES SlaveAxisID = AXISNAMES::X_1;
		/// <summary>
		/// ID of the Cam
		/// </summary>
		PMC_USINT CamID = 128;

		/// <summary>
		/// operating mode of the cam, can set cam to automatically repeat
		/// </summary>
		CAMMODE CamMode = CAMMODE::AutoStart;

		/// <summary>
		/// Scaling factor for master axis position interval
		/// </summary>
		PMC_REAL MasterAxisScaling = 1;

		/// <summary>
		/// Scaling factor for slave axis position values
		/// </summary>
		PMC_REAL SlaveAxisScaling = 1;

		/// <summary>
		/// The offset between the start of master axis and the specified origin point, in meters
		/// </summary>
		PMC_REAL MasterAxisOffsetM = 0;

		/// <summary>
		/// In absolute mode, the origin position of the cam master axis is the same as the system origin. In relative mode, the origin position of the cam master axis is the same as the current position of the master xbot when the cam was activated
		/// </summary>
		POSITIONMODE MasterAxisOffsetMode = POSITIONMODE::RELATIVE_MOTION;

		/// <summary>
		/// The offset between the start of slave axis and the specified origin point, in meters
		/// </summary>
		PMC_REAL SlaveAxisOffsetM = 0;

		/// <summary>
		/// In absolute mode, the origin position of the cam slave axis is the same as the system origin. In relative mode, the origin position of the cam slave axis is the same as the current position of the slave xbot when the cam was activated
		/// </summary>
		POSITIONMODE SlaveAxisOffsetMode = POSITIONMODE::RELATIVE_MOTION;

		/// <summary>
		/// ratchet direction of the cam master axis. choose whether cam is engage whether the master cam position is increasing or decreasing, only engaged when it's increasing, or only when it's decreasing
		/// </summary>
		CAMRATCHETDIRECTION MasterAxisRatchetDirection;
	};

	/// <summary>
	/// Define the queue region
	/// </summary>
	struct QueueDefinition
	{
		/// <summary>
		/// starting x coordinate of the queue area, in meters
		/// </summary>
		PMC_REAL AreaLeftXm = 0;
		/// <summary>
		/// starting y coordinate of the queue area, in meters
		/// </summary>
		PMC_REAL AreaBottomYm = 0;
		/// <summary>
		/// ending x coordinate of the queue area, in meters
		/// </summary>
		PMC_REAL AreaRightXm = 0;
		/// <summary>
		/// ending y coordinate of the queue area, in meters
		/// </summary>
		PMC_REAL AreaTopYm = 0;
		/// <summary>
		/// entry area of the queue region
		/// </summary>
		AREASELECTION EntryLocation;
		/// <summary>
		/// exit area of the queue region
		/// </summary>
		AREASELECTION ExitLocation;
		/// <summary>
		/// size of the largest Xbot allowed in the queue, x dimension, in meters
		/// </summary>
		PMC_REAL MaxXbotXSizem = 0;
		/// <summary>
		/// size of the largest xbot allowed in the queue, y dimension, in meters
		/// </summary>
		PMC_REAL MaxXbotYSizem = 0;
		/// <summary>
		/// max speed used by the queue management to move the XBots
		/// </summary>
		PMC_REAL MaxSpeedInQueue = 0;
		/// <summary>
		/// max acceleration used by the queue management to move the XBots
		/// </summary>
		PMC_REAL MaxAccelInQueue = 0;
	};

	/// <summary>
	/// Define the auto load region
	/// </summary>
	struct AutoLoadZoneDefinition
	{
		/// <summary>
		/// center of the zone at the boundary of the flyway, x position, meters
		/// </summary>
		PMC_REAL BoundaryCenterXm = 0;
		/// <summary>
		/// center of the zone at the boundary of the flyway, y position, meters
		/// </summary>
		PMC_REAL BoundaryCenterYm = 0;
		/// <summary>
		/// Length of the zone, meters
		/// </summary>
		PMC_REAL ZoneLengthm = 0;
		/// <summary>
		/// Width of the zone, meters
		/// </summary>
		PMC_REAL ZoneWidthm = 0;
		/// <summary>
		/// loading zone = adding XBot to Flyway. unloading zone = removing XBot from flyway
		/// </summary>
		ALZONETYPE ZoneType;
		/// <summary>
		/// stop before unloading, or unload without stopping
		/// </summary>
		ALZONEUNLOADMODE UnloadingMode;
		/// <summary>
		/// x size of the Xbot to load / unload, meters
		/// </summary>
		PMC_REAL MaxXbotXSizem = 0;
		/// <summary>
		/// y size of the Xbot to load / unload, meters
		/// </summary>
		PMC_REAL MaxXbotYSizem = 0;
		/// <summary>
		/// max speed used to load or unload XBots
		/// </summary>
		PMC_REAL MaxSpeed = 0;
		/// <summary>
		/// max acceleration used to load or unload XBots
		/// </summary>
		PMC_REAL MaxAccel = 0;
	};

	/// <summary>
	/// Basic Queue Status information
	/// </summary>
	struct QueueStatus
	{
		/// <summary>
		/// The entry area is empty and ready to receive an XBot
		/// </summary>
		PMC_BOOL EntryReadyToReceiveXBot;
		/// <summary>
		/// there is an XBot available at the exit location
		/// </summary>
		PMC_BOOL XBotAvailableAtExit;
		/// <summary>
		/// ID of the XBot available at the exit location
		/// </summary>
		PMC_USINT XBotIDAtExit = 0;
		/// <summary>
		/// center x position of the entry area of the queue
		/// </summary>
		PMC_REAL QueueEntryXposm = 0;
		/// <summary>
		/// center y position of the entry area of the queue
		/// </summary>
		PMC_REAL QueueEntryYposm = 0;
	};

	/// <summary>
	/// the return value for the GetQueueStatus Command
	/// </summary>
	struct QueueStatusReturn
	{
		/// <summary>
		/// return value for checking the queue status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// the status of the queue
		/// </summary>
		QueueStatus queueStatus;
	};

	/// <summary>
	/// information about the queue
	/// </summary>
	struct QueueInfo
	{
		/// <summary>
		/// The entry area is empty and ready to receive an XBot
		/// </summary>
		PMC_BOOL EntryReadyToReceiveXBot;
		/// <summary>
		/// there is an XBot available at the exit location
		/// </summary>
		PMC_BOOL XBotAvailableAtExit;
		/// <summary>
		/// number of XBots being managed by the Queue
		/// </summary>
		PMC_USINT XBotCount = 0;
		/// <summary>
		/// Array of XBot IDs of the XBots being managed by the Queue
		/// </summary>
		PMC_USINT QueuedXBotsIDs[100];
	};

	/// <summary>
	/// the return value for the GetQueueInfo Command
	/// </summary>
	class QueueInfoReturn
	{
		/// <summary>
		/// return value for checking the queue info, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// detailed info of the queue
		/// </summary>
		QueueInfo queueInfo;
	};

	/// the return value for the AutoLoadingZone_GetAllStatus Command
	struct AutoLoadZoneAllStatusReturn
	{
		/// <summary>
		/// return value for checking all autoloadingzone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;		
		/// <summary>
		/// Number of autoloading zones
		/// </summary>
		PMC_USINT nZones = 0;		
		/// <summary>
		/// The IDs of the autoloading zones
		/// </summary>
		PMC_USINT ZoneIDs[40];
		/// <summary>
		/// 0: undefined, 1: defined, 2: unloading zone, 3: loading zone
		/// </summary>
		PMC_USINT ZoneStates[40];
		/// <summary>
		/// 0: busy, 1: ready (if loading zone ready = xbot ready to fetch,if unloading zone ready = zone ready to receive new xbot)
		/// </summary>
		PMC_USINT EntranceExitStates[40];
		/// <summary>
		/// xbot ID added to unloading zone or ready to fetch from loading zone 
		/// </summary>
		PMC_USINT XbotIDs[40];
		/// <summary>
		/// zone error code
		/// </summary>
		PMC_USINT LoadingErrorCode[40];
	};

	/// <summary>
	/// the return value for the AutoLoadingZone_GetZoneStatus Command
	/// </summary>
	struct AutoLoadZoneStatusReturn
	{
		/// <summary>
		/// return value for checking the zone status, the validity of the remaining data in this struct in verified by this field
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// whether the zone is undefined, unloading, or loading state
		/// </summary>
		ALZONESTATE ZoneState;
		/// <summary>
		/// ID of the XBot loaded at the loading zone
		/// </summary>
		PMC_USINT LoadedXBotID = 0;
		/// <summary>
		/// unloading zone ready for next XBot means it is ready to receive another XBot to send to the external device
		/// loading zone ready for next XBot means it is ready to receive another XBot from the external device
		/// </summary>
		PMC_BOOL ReadyForNextXBot;
		/// <summary>
		/// number of XBots inside the zone
		/// </summary>
		PMC_USINT XBotCount = 0;
		/// <summary>
		/// x position of the unloading point in an unloading zone
		/// </summary>
		PMC_REAL UnloadingEntryXposm = 0;
		/// <summary>
		/// y position of the unloading point in an unloading zone
		/// </summary>
		PMC_REAL UnloadingEntryYposm = 0;
	};

	/// <summary>
	/// Define the star wheel
	/// </summary>
	struct StarWheelDefinition
	{
		/// <summary>
		/// Disc radius in meters
		/// </summary>
		PMC_REAL DiscRadius = 0;
		/// <summary>
		/// Max disc rotation speed in radians/sec (must be positive)
		/// </summary>
		PMC_REAL MaxDiscSpeed = 0;
		/// <summary>
		/// Max xbot acceleration in meters/sec^2 (must be positive)
		/// </summary>
		PMC_REAL MaxXbotAcc = 0;
		/// <summary>
		/// Start of sync section in radians (if CCW must be less than end angle, if CW must be greater than end angle) 
		/// </summary>
		PMC_REAL SyncAngleBegin = 0;
		/// <summary>
		/// End of sync section in ratdians (if CCW must be greater than end angle, if CW must be less than end angle)
		/// </summary>
		PMC_REAL SyncAngleEnd = 0;
		/// <summary>
		/// X coordinate of disc center in meters
		/// </summary>
		PMC_REAL DiscCenterX = 0;
		/// <summary>
		/// Y coordinate of disc center in meters
		/// </summary>
		PMC_REAL DiscCenterY = 0;
		/// <summary>
		/// X coordinate of starting location in meters
		/// </summary>
		PMC_REAL StartLocationX = 0;
		/// <summary>
		/// Y coordinate of starting location in meters
		/// </summary>
		PMC_REAL StartLocationY = 0;
		/// <summary>
		/// X coordinate of ending location in meters
		/// </summary>
		PMC_REAL EndLocationX = 0;
		/// <summary>
		/// Y coordinate of ending location in meters
		/// </summary>
		PMC_REAL EndLocationY = 0;
		/// <summary>
		/// locations of vials on the disc in radians (must be 0-2PI);
		/// </summary>
		PMC_REAL VialLocations[100];
		/// <summary>
		/// number of vials on the disc (less than or equal to 100)
		/// </summary>
		PMC_USINT NumVials = 0;
		/// <summary>
		/// Direction of disc rotation (0 = clockwise,1 = counterclockwise)
		/// </summary>
		PMC_USINT Direction = 0;
		/// <summary>
		/// ID of master xbot (must be virtual 100-127)
		/// </summary>
		PMC_USINT MasterXID = 0;
	};

	/// <summary>
	/// Status of star wheel
	/// </summary>
	struct StarWheelStatus
	{
		/// <summary>
		/// Star wheel state: 0 = unassigned, 1 = assigned
		/// </summary>
		PMC_USINT State = 0;
		/// <summary>
		/// Start location X coordinate in meters
		/// </summary>
		PMC_REAL StartLocationX = 0;
		/// <summary>
		/// Start location Y coordinate in meters
		/// </summary>
		PMC_REAL StartLocationY = 0;
		/// <summary>
		/// End location X coordinate in meters
		/// </summary>
		PMC_REAL EndLocationX = 0;
		/// <summary>
		/// End location Y coordinate in meters
		/// </summary>
		PMC_REAL EndLocationY = 0;
		/// <summary>
		/// Entrance status: 0 - not ready to receive new xbot, 1 - ready to receive new xbot
		/// </summary>
		PMC_USINT EntranceStatus = 0;
		/// <summary>
		/// Exit status: 0 - xbot is not ready to fetch, 1 - xbot is ready to fetch
		/// </summary>
		PMC_USINT ExitStatus = 0;
		/// <summary>
		/// Xbot ID to fetch 
		/// </summary>
		PMC_USINT ExitXID = 0;
	};

	/// <summary>
	/// Star wheel status return wrapper
	/// </summary>	
	struct StarWheelStatusRtn
	{
		/// <summary>
		/// Return code
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// Star wheel status
		/// </summary>
		StarWheelStatus WheelStatus;
	};

	struct GCodeRtn
	{
		/// <summary>
		/// Return code
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// length of GCode file in bytes
		/// </summary>
		PMC_DINT GCodeLength = 0;
		/// <summary>
		/// pointer to the raw binary of GCode file 
		/// </summary>
		PMC_BYTE GCodeFile[200000];
	};

	struct ConfigRtn
	{
		/// <summary>
		/// Return code
		/// </summary>
		PMCRTN PmcRtn = PMCRTN::ALLOK;
		/// <summary>
		/// length of config file in bytes
		/// </summary>
		PMC_DINT ConfigLength = 0;
		/// <summary>
		/// pointer to the raw binary of config file 
		/// </summary>
		PMC_BYTE ConfigFile[500000];
	};

	struct GCodeIDRtn
	{				
		/// <summary>
		/// Return code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of G-code files on the PMC
		/// </summary>
		PMC_USINT nIDs;		
		/// <summary>
		/// the IDs of the G-code files on the PMC
		/// </summary>
		PMC_USINT GCodeIDs[64];
	};

	struct AccidentXbotRtn
	{		
		/// <summary>
		/// Return code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// The number of accident xbots
		/// </summary>
		PMC_USINT nXbots;		
		/// <summary>
		/// The accident xbot IDs
		/// </summary>
		PMC_USINT XbotIDs[150];
	};

	struct LogRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of bytes in the log file
		/// </summary>
		PMC_DINT nBytes;		
		/// <summary>
		/// the log file as a raw binary
		/// </summary>
		PMC_BYTE Bytes[100000];
	};

	struct SerialNumRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// The 64-bit serial number
		/// </summary>
		PMC_ULINT SerialNum;		
		/// <summary>
		/// Flyway firmware version number
		/// </summary>
		PMC_UDINT FirmwareVersion;		
		/// <summary>
		/// Application data version
		/// </summary>
		PMC_UDINT AppDataVersion;
	};

	struct VersionRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;
		/// <summary>
		/// The 64-bit version number
		/// </summary>
		PMC_ULINT VersionNum;
	};

	struct TrajectoryHashRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// what axises are in the trajectory file: bit0=X; bit1=Y; bit2=Z; bit3=Rx; bit4=Ry; bit5=Rz; bit6=DO (1 means axis selected; 0 means not selected)
		/// </summary>
		PMC_USINT Axises = 0;		
		/// <summary>
		/// time interval between positions (seconds)  
		/// </summary>
		PMC_REAL TimeInterval = 0;		
		/// <summary>
		/// MD5 hash of the traj file
		/// </summary>
		PMC_USINT TrajHash[16];
	};

	struct FlywayTempRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Flyway power consumption in watts
		/// </summary>
		PMC_REAL Power;		
		/// <summary>
		/// Flyway CPU temperature in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL CPUTemp;		
		/// <summary>
		/// Power amplifier temperature in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL PATemp;		
		/// <summary>
		/// Motor temperature in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL MTemp;		
		/// <summary>
		/// Flyway CPU temperature in degrees Celsius (floating point precision)
		/// </summary>
		PMC_REAL CPUTempFloat;		
		/// <summary>
		/// Power amplifier temperature in degrees Celsius (floating point precision)
		/// </summary>
		PMC_REAL PATempFloat;		
		/// <summary>
		/// Motor temperature in degrees Celsius (floating point precision)
		/// </summary>
		PMC_REAL MTempFloat;
	};

	struct FlywayErrorRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// The read error code (0 = no error)
		/// </summary>
		PMC_UDINT FlywayErrorCode;
	};

	struct IncomingXbotsRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of incoming xbots
		/// </summary>
		PMC_USINT nXbots;				
		/// <summary>
		/// The IDs of the incoming xbots
		/// </summary>
		PMC_USINT XbotIDs[127];
		/// <summary>
		/// The IDs of the borders that the xbots are crossing
		/// </summary>
		PMC_USINT BorderIDs[127];
	};

	struct ErrorCodeRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of error codes read 
		/// </summary>
		PMC_USINT NumErrors;		
		/// <summary>
		/// The read error codes
		/// </summary>
		PMC_DINT ErrorCodes[9];
	};

	struct PMCStatusRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// PMC Statue
		/// </summary>
		PMCSTATUS State;		
		/// <summary>
		/// number of XBots in the system
		/// </summary>
		PMC_UINT nXbots;		
		/// <summary>
		/// system power comsumption in Watts
		/// </summary>
		PMC_REAL Power;		
		/// <summary>
		/// highest cpu temperature in the system in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL MaxCPUTemp;		
		/// <summary>
		/// highest amplifier temperature in the system in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL MaxPATemp;		
		/// <summary>
		/// highest motor temperature in the system in degrees Celsius (integer precision)
		/// </summary>
		PMC_REAL MaxMTemp;		
		/// <summary>
		/// 0: connected and synced; 1: disconnected; 2: connected but not synced
		/// </summary>
		PMC_USINT FieldbusStatus;		
		/// <summary>
		/// 0: connected; 1: disconnected
		/// </summary>
		PMC_USINT PMNetStatus;		
		/// <summary>
		/// 0: at least one xbot doesn't finish XID scan; 1: all xbots finished XID scan
		/// </summary>
		PMC_USINT MoverIDScanStatus;		
		/// <summary>
		/// current PMC error code (0 is OK)
		/// </summary>
		PMC_UDINT PMCErrorCode;
	};

	struct DigitalSignalRtn
	{		
		/// <summary>
		/// Digital signal was read successfully
		/// </summary>
		PMC_BOOL valid = false;		
		/// <summary>
		/// level of the digital signal
		/// </summary>
		PMC_BOOL level;
	};

	struct ReadNumXbotsRtn
	{		
		/// <summary>
		/// number of xbots was read successfully
		/// </summary>
		PMC_BOOL valid = false;		
		/// <summary>
		/// The number of xbots
		/// </summary>
		PMC_USINT nXbots = 0;
	};

	struct PMCStateRtn
	{		
		/// <summary>
		/// successfully read PMC state
		/// </summary>
		PMC_BOOL valid = false;		
		/// <summary>
		/// 0: booting; 1: inactive; 2: activating; 5: operation; 7: deactivating; 8: error handling; 9: error
		/// </summary>
		PMCSTATUS state;
	};

	struct FeedbackRtn
	{		
		/// <summary>
		/// successfully read xbot stream feedback
		/// </summary>
		PMC_BOOL valid = false;		
		/// <summary>
		/// value read from stream: meters/Newtons for X, Y, Z; radians/Newton*Meters for Rx, Ry, Rz
		/// </summary>
		PMC_REAL FeedbackValue = 0;
	};

	struct ReadXbotIDsRtn
	{		
		/// <summary>
		/// successfully read xbot IDs
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// number of xbots in system
		/// </summary>
		PMC_USINT NumXbots;		
		/// <summary>
		/// xbot ID#s of xbots in the system
		/// </summary>
		PMC_USINT Xbot_IDs[127];
	};

	struct GetXbotIDsRtn
	{		
		/// <summary>
		/// successfully read xbot IDs
		/// </summary>
		PMC_BOOL done;		
		/// <summary>
		/// number of xbots in system
		/// </summary>
		PMC_USINT NumXbots;		
		/// <summary>
		/// xbot ID#s of xbots in the system
		/// </summary>
		PMC_USINT Xbot_IDs[127];
	};

	struct ReadXbotPosRtn
	{		
		/// <summary>
		/// xbot position was read successfully 
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// xbot X position in meters
		/// </summary>
		PMC_REAL PosX;		
		/// <summary>
		/// xbot Y position in meters
		/// </summary>
		PMC_REAL PosY;		
		/// <summary>
		/// xbot Z position in meters
		/// </summary>
		PMC_REAL PosZ;		
		/// <summary>
		/// xbot rotation about X axis in radians 
		/// </summary>
		PMC_REAL PosRx;		
		/// <summary>
		/// xbot rotation about Y axis in radians 
		/// </summary>
		PMC_REAL PosRy;		
		/// <summary>
		/// xbot rotation about Z axis in radians 
		/// </summary>
		PMC_REAL PosRz;
	};

	struct ReadXbotStateRtn
	{		
		/// <summary>
		/// successfully read xbot state
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// 0 = undetected, 1 = discovering, 2 = landed, 3 = idle, 4 = disabled, 5 = in motion, 6 = waiting, 7 = stopping, 8 = obstacle, 9 = hold, 10 = stopped, 14 = error
		/// </summary>
		XBOTSTATE State;
	};

	struct ReadXbotStatusRtn
	{		
		/// <summary>
		/// xbot status is up-to-date, if false the output position is the last known status
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// 0: position mode; 1: force mode
		/// </summary>
		PMC_USINT ControlMode;		
		/// <summary>
		/// 0: resumed; 1:paused
		/// </summary>
		PMC_USINT MotionStatus;
		/// <summary>
		/// 0: buffer released; 1:buffer blocked
		/// </summary>
		PMC_USINT BufferStatus;
		/// <summary>
		/// 0: unbond to group; 1:bonded to group
		/// </summary>
		PMC_USINT GroupStatus;
		/// <summary>
		/// 0: unlinked to planet; 1: linked to planet
		/// </summary>
		PMC_USINT PlanetStatus;
		/// <summary>
		/// 0: xbot free; 1: xbot locked
		/// </summary>
		PMC_USINT LockStatus;
		/// <summary>
		/// active stereotype ID for this xbot 
		/// </summary>
		PMC_USINT StereotypeID;
	};

	struct FunctionRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// The assigned function id number
		/// </summary>
		PMC_USINT FunctionID;
	};

	struct FunctionInfoRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;				
		/// <summary>
		/// The function state: 0 - inactive, 1 - active
		/// </summary>
		PMC_USINT functionState;
		/// <summary>
		/// The number of commands in the function
		/// </summary>
		PMC_USINT numCmds;
		/// <summary>
		/// The number xbots in the function
		/// </summary>
		PMC_USINT numXbots;
	};

	struct StartUpPMSRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;
		/// <summary>
		/// number of xbots in system
		/// </summary>
		PMC_USINT NumXbots = 0;
		/// <summary>
		/// xbot ID#s of xbots in the system
		/// </summary>
		PMC_USINT Xbot_IDs[127];
	};

	struct ScanXIDRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// 1 : valid RFID detected, 0 : invalid
		/// </summary>
		PMC_USINT Status;		
		/// <summary>
		/// xbot global ID 
		/// </summary>
		PMC_ULINT XID;
	};

	struct ReadTimeStampRtn
	{
		/// <summary>
		/// successfully read the timestamp 
		/// </summary>
		PMC_BOOL valid;
		/// <summary>
		/// timestamp value in nanoseconds (local PMC time since power on) 
		/// </summary>
		PMC_ULINT TimeStamp_ns;
	};
	
	struct ReadXbotSpeedRtn
	{		
		/// <summary>
		/// successfully read the xbot speed
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// speed in the X axis in m/s
		/// </summary>
		PMC_REAL SpeedX;		
		/// <summary>
		/// speed in the Y axis in m/s
		/// </summary>
		PMC_REAL SpeedY;		
		/// <summary>
		/// speed in the Rz axis in m/s
		/// </summary>
		PMC_REAL SpeedRz;
	};

	struct Sector_GetStatusRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;
		/// <summary>
		/// 0: not defined, 1: disconnected, 2: inactive and fenced, 3: deactivating, 4: stopping, 5: activating, 6: activated and fenced, 7: in operation
		/// </summary>
		PMC_USINT State;		
		/// <summary>
		/// Number of xbots in the sector
		/// </summary>
		PMC_USINT NumXbots;		
		/// <summary>
		/// IDs of the xbots in the sector
		/// </summary>
		PMC_USINT XbotIDs[32];
	};

	struct ThroughputTestRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Full duplex 1 ms fieldbus throughput
		/// </summary>
		PMC_UDINT Throughput1ms;
	};

	struct GetXbotRFIDsRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of RFIDs scanned
		/// </summary>
		PMC_USINT NumRFIDs;		
		/// <summary>
		/// the 64 bits scannded RFIDs
		/// </summary>
		PMC_ULINT RFIDs[127];			
		/// <summary>
		/// corresponding xbot IDs
		/// </summary>
		PMC_USINT XbotIDs[127];
	};

	struct LinkedMover_AssignRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// linked xbot ID, used to drive the linked xbots
		/// </summary>
		PMC_USINT LinkedXbotID;
	};

	struct FindXbotsInAreaRtn
	{
		/// <summary>
		/// 1 if values are valid
		/// </summary>
		PMC_BOOL valid;		
		/// <summary>
		/// Number of xbots whose centers are in the area
		/// </summary>
		PMC_USINT nXbots;		
		/// <summary>
		/// IDs of xbots whose centers are in the area
		/// </summary>
		PMC_USINT XbotIDs[127];
	};

	struct GetSpacedXbotPositionsRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;
		/// <summary>
		/// Number xbots in system
		/// </summary>
		PMC_USINT nXbots;
		/// <summary>
		/// The IDs of the xbots in the system
		/// </summary>
		PMC_USINT XbotIDs[127];
		/// <summary>
		/// Spaced out X positions
		/// </summary>
		PMC_REAL PosX[127];
		/// <summary>
		/// Spaced out Y positions
		/// </summary>
		PMC_REAL PosY[127];
	};

	struct App_GetAllStationXbotsRtn
	{
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Total number of bays in the system
		/// </summary>
		PMC_UINT NumBays;			
		/// <summary>
		/// IDs of the station
		/// </summary>
		PMC_UINT StationIDs[5000];
		/// <summary>
		/// IDs of the bay
		/// </summary>
		PMC_USINT BayIDs[5000];
		/// <summary>
		/// IDs of the xbot
		/// </summary>
		PMC_USINT XbotIDs[5000];
	};

	struct App_GetBayXbotIDRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Id of the xbot in the station's bay
		/// </summary>
		PMC_USINT XbotID;
	};

	struct App_GetDesignRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Number of bytes in the read configuration file
		/// </summary>
		PMC_DINT nBytes;		
		/// <summary>
		/// the application designer file as a raw binary
		/// </summary>
		PMC_BYTE Bytes[500000];
	};

	struct App_GetStationXbotIDsRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// number of bays in the station
		/// </summary>
		PMC_USINT NumBays;		
		/// <summary>
		/// IDs of the xbots in the station's bays
		/// </summary>
		PMC_USINT XbotIDs[30];
	};

	struct App_GetXbotTargetRtn
	{		
		/// <summary>
		/// Return Code
		/// </summary>
		PMCRTN PmcRtn;		
		/// <summary>
		/// Station ID (>0)
		/// </summary>
		PMC_USINT StationID;		
		/// <summary>
		/// Bay ID (>0)
		/// </summary>
		PMC_USINT BayID;
	};

	struct MoverID_GetScanLocationRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// X coordinate of the bottom left corner of the new station (m)
		PMC_REAL ScanLocationX;
		/// Y coordinate of the bottom left corner of the new station (m)
		PMC_REAL ScanLocationY;
	};

	struct MoverID_ReadScanStatusRtn
	{
		/// xbot state is up-to-date, if false the output state is the last known state
		PMC_BOOL Valid;
		/// false: unscanned, true: scanned
		PMC_BOOL ScanStatus;
	};

	struct MoverID_ScanMoverIDRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// 1 : valid RFID detected, 0 : invalid
		PMC_USINT Status;
		/// xbot global ID 
		PMC_ULINT XID;
	};

	struct MoverLabel_GetLabelRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Mover label assigned to the xbot
		PMC_UINT MoverLabel;
	};

	struct MoverLabel_GetXbotIDRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Xbot ID with the mover label
		PMC_USINT XbotID;
	};

	struct StreamModeCtrlRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Starting X positions of the xbots 
		PMC_REAL StreamStartPosX[9];
		/// Starting Y positions of the xbots 
		PMC_REAL StreamStartPosY[9];
	};

	struct GetAllAccidentLocationsRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Number of accidents
		PMC_USINT nAccidents;
		/// bottom left corner X coordinate of accident zone
		PMC_REAL AccidentBottomLeftX[225];
		/// bottom left corner Y coordinate of accident zone
		PMC_REAL AccidentBottomLeftY[225];
		/// top right corner X coordinate of accident zone
		PMC_REAL AccidentTopRightX[225];
		/// top right corner Y coordinate of accident zone
		PMC_REAL AccidentTopRightY[225];
	};

	struct GetFlywayConnectErrorCountsRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Number of flyways in the system
		PMC_USINT nFlyways;
		/// The status of each flyway
		PMC_USINT FlywayStatus[200];
		/// The number of flyway connection errors for each flyway
		PMC_UDINT ConnectionErrorCount[200];
	};

	struct GetPMNetErrorCountsRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Number of flyways in the system
		PMC_USINT nFlyways;
		/// Number of PMNet errors on the north edge
		PMC_USINT FlywayNorthConnectionErrorCount[200];
		/// Number of PMNet errors on the east edge
		PMC_USINT FlywayEastConnectionErrorCount[200];
		/// Number of PMNet errors on the south edge
		PMC_USINT FlywaySouthConnectionErrorCount[200];
		/// Number of PMNet errors on the west edge
		PMC_USINT FlywayWestConnectionErrorCount[200];
	};

	struct ReadXbotForceRtn
	{
		/// xbot force was read successfully
		PMC_BOOL Valid;
		/// force in X in Newtons
		PMC_REAL Fx;
		/// force in Y in Newtons
		PMC_REAL Fy;
		/// force in Z in Newtons 
		PMC_REAL Fz;
		/// torque about X axis in Newton*Meters
		PMC_REAL Tx;
		/// torque about Y axis in Newton*Meters
		PMC_REAL Ty;
		/// torque about Z axis in Newton*Meters
		PMC_REAL Tz;
	};

	struct IncomingXbotsExRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// Number of incoming xbots
		PMC_USINT nXbots;
		/// The IDs of the incoming xbots
		PMC_USINT XbotIDs[127];
		/// The IDs of the borders that the xbots are crossing
		PMC_USINT BorderIDs[127];
		/// the mover types of the xbots that are crossing
		PMC_USINT MoverTypes[127];
		/// mover labels of the xbots that are crossing 
		PMC_UINT MoverLabels[127];
		/// the stereotype IDs of the xbots that are crossing 
		PMC_USINT StereotypeIDs[127];
	};

	struct GetPlanetsRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// number of planet xbots attached to the star xbot
		PMC_USINT NumPlanets;
		/// IDs of planet xbots attached
		PMC_USINT PlanetXbotIDs[32];
	};

	struct GetStarRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// 0: Not linked to any xbot; >0: star Xbot ID
		PMC_USINT StarXbotID;
	};

	struct GetFlywayPositionRtn
	{
		/// Return Code
		PMCRTN PmcRtn;
		/// X position of the bottom left corner of flyway (m)
		PMC_REAL PosX;
		///Y position of the bottom left corner of flyway (m)
		PMC_REAL PosY;
	};
};